/*
 * Copyright (c) 2016, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
 * \file :   aic3104_test.c
 *
 * \brief:   Routines to test the Audio Codec on Base EVM
 *
 * \b Desc:  This file contains the routines for configuring and
 * \n        testing the aic3104 Audio codec on the Base Board.
 *
 */

/*
 *====================
 * Includes
 *====================
 */
#include "aic3104.h"
#include "stdio.h"
#include "dm8127_i2c.h"
#include "dm8127_Platform.h"
#include "dm8127_evm.h"
 
/*
 *====================
 * Extern declarations
 *====================
 */
/***
 * \brief: Routine to configure the Headphone Out and play sine tone
 *
 * \b Descr: This routine initializes the aic3104 Audio Codec on the
 * \n       base board and plays a sine tone on the head-phone.
 *
 * \param : u8McaspNum	[IN]	Instance of McASP number
 *
 * \return:  SUCCESS  on success
 * \n        FAILED   if any error
 */

extern STATUS aic3104_headphone_out
(
	UINT8 u8McaspNum
);

extern STATUS aic3104_lineout_out
(
	UINT8 u8McaspNum
);

extern STATUS aic3104_MicHeadphone_Loop
(
	UINT8 u8McaspNum
);

extern STATUS aic3104_LineInLineOut_Loop
(
	UINT8 u8McaspNum
);


/*
 *=========================
 * Function implementations
 *=========================
 */

/***
 * \brief: Routine to test the aic3104 Audio codec on Base board
 *
 * \b Descr: This routine initializes the aic3104 Audio Codec on the
 * \n        base board to test various interfaces including the
 * \n        following.
 *
 * \n      (a) Playback on Head-phohne
 * \n      (b) Playback on Line-Out
 * \n      (c) Loopback the input on Microphone to head-set.
 * \n      (d) Loopback the input on Line-In to Line-Out
 *
 * \param: u8McaspNum	[IN]	Instance of McASP number
 *
 * \return   SUCCESS    success
 * \n        FAILED     error
 */
STATUS aic3104_tests
(
	UINT8    u8McaspNum
)
{
	/* begin aic3104_tests */
	STATUS u32RetVal = SUCCESS;
	
	/* Initialize the I2C device on BUS No ..... This will initialize the slave
	 * having Address 0x18 */

	u32RetVal = DM8127I2cInit(aic3104_I2C_BUS_NUM_2);
	
	if (SUCCESS != u32RetVal)
	{
#ifdef DEBUG
		printf ("I2C initialization failed for buss number %d.\r\n", aic3104_I2C_BUS_NUM_2);
#endif
		return FAILED;
	}


#ifdef DEBUG

	printf ("+=============================================================+\r\n"
			"|*********  AUDIO CODEC AIC3104 TEST ON BASE BOARD  **********|\r\n"
			"|  This test covers connectivity tests to AIC3104 on the base |\r\n"
			"|  for head-phone, Line-in. For this                          |\r\n"
			"|  ensure the following :-                                    |\r\n"
			"|                                                             |\r\n"
			"|      (a) Head phone connected to the Head-phone jack.       |\r\n"
			"|      (b) Stereo Audio cable feeding into the Line-In jack.  |\r\n"
			"|                                                             |\r\n"
			"| Playing the sine tone on the Head phone                     |\r\n"
			"| Look for the tone on the Head phone for the next 5 seconds. |\r\n"
			"+=============================================================+\r\n");
	printf (" Playing the sine tone the Line-out. \r\n");
	printf (" Look for tone being played back on Speakers for next 5 seconds \r\n");
#else
	Uart_stringSend("\r\n+=============================================================+"
			        "\r\n|*********  AUDIO CODEC AIC3104 TEST ON BASE BOARD  **********|"
			        "\r\n|  This test covers connectivity tests to AIC3104 on the base |"
		          	"\r\n|  for head-phone, Line-in. For this                          |"
			        "\r\n|  ensure the following :-                                    |"
			        "\r\n|                                                             |"
			        "\r\n|      (a) Head phone connected to the Head-phone jack.       |"
			        "\r\n|      (b) Stereo Audio cable feeding into the Line-In jack.  |"
			        "\r\n|                                                             |"
#ifndef  PRE_COMPLIANCE
			        "\r\n| Playing the sine tone on the Head phone                     |"
#endif
			        "\r\n| Look for the tone on the Head phone for the next 5 seconds. |"
			        "\r\n+=============================================================+\r\n\n");


#endif

#ifndef PRE_COMPLIANCE

	Uart_stringSend ("Playing the sine tone the Line-out. \r\n"
					 "Look for tone being played back on Speakers for next 5 seconds\r\n");
	u32RetVal = aic3104_headphone_out (u8McaspNum);
	if (FAILED == u32RetVal)
	{
#ifdef DEBUG
		printf (" Error occured during the configuraiton of aic3104.\r\n");
#else
		Uart_stringSend(" Error occured during the configuraiton of aic3104.\r\n");
#endif
		return (u32RetVal);
	}

#endif


#ifdef LINE_OUT

	u32RetVal = aic3104_lineout_out (u8McaspNum);
	if (FAILED == u32RetVal)
	{
		printf (" Error occured during the configuraiton of aic3104.\r\n");
		return (u32RetVal) ;
	}

	printf (" Loopback of Microphone to the headphone. \r\n");
	printf (" Speak on the microphone and hear back on the head phone. \r\n");

#endif

#ifdef DEBUG
	printf (" Loopback of Line-In to the Line-Out. \r\n");
	printf (" Feed the stereo audio into the Line in and hear back on speaker. \r\n");
#else
	Uart_stringSend ("\n\rLoopback of Line-In to the Line-Out. \r\n");
	Uart_stringSend ("\rFeed the stereo audio into the Line in and hear back on speaker. \r\n\n");
#endif

	u32RetVal = aic3104_MicHeadphone_Loop (u8McaspNum);
	if (FAILED == u32RetVal)
	{
#ifdef DEBUG
		printf (" Error occured during the configuraiton of aic3104.\r\n");
#else
		Uart_stringSend(" Error occured during the configuraiton of aic3104.\r\n");
#endif
		return (u32RetVal);
	}

#ifdef  LINE_IN_OUT
	u32RetVal = aic3104_LineInLineOut_Loop (u8McaspNum);
	if (FAILED == u32RetVal)
	{
		printf (" Error occured during the configuraiton of aic3104.\r\n");
		return (u32RetVal);
	}
#endif
	return (u32RetVal);
}	/* end aic3104_tests */


/***
 * \brief:   Routine to test the AIC3104 Audio codec on Base board
 *
 * \b Descr: This routine initializes the AIC3104 Audio Codec on the base board
 * \n        to test various interfaces including the following.
 *
 *	\n      (a) Playback on Head-phohne
 *  \n      (b) Playback on Line-Out
 *  \n      (c) Loopback the input on Microphone to head-set.
 *  \n      (d) Loopback the input on Line-In to Line-Out
 *
 * \param:  u8McaspNum	[IN]	Instance of McASP number
 *
 * \return:    SUCCESS for success  - Description
 * \n 	       FAILED for error   - Description
 *
 */
STATUS Audio_Test
(
	void *testargs
)
{
	STATUS u32RetVal = FAILED;

//	DM8127_EVM_Init ();
	  Uart_stringSend("\r\n==================================="
					  "\r\n            Audio Test             "
					  "\r\n===================================\r\n");
	u32RetVal = aic3104_tests (2);

	if (u32RetVal == SUCCESS)
	{
#ifdef DEBUG
		printf (" All configuration & Playback/loopback successfully completed.\r\n"
				" However, actual results are to be judged based on the audio \r\n"
				" heard from the head-phone and speakers.\r\n");
#else
		Uart_stringSend ("All configuration & Playback/loopback successfully"
				         " completed.\r\nHowever, actual results are to be "
				         " judged based on the audio \r\nheard from the head"
				         "-phone and speakers.\r\n\n");
#endif
	}
	else
	{
#ifdef DEBUG
		printf ("Some configuration failed.\r\n");
#else
		Uart_stringSend("\r\n Failed to configured the Audio codec\r\n");
#endif
	}
	Uart_stringSend("\r\n-------------------------X-----------------------------");
	return(u32RetVal);
}

